﻿//*********************************************************
//  C# API - file for MEGSV86x64.DLL (Windows 64-Bit)
//  For using with GSV-8/BX8 or GSV-6/ITA measuring amplifiers with serial interface
//  Copyright (c) 2019 ME-Messysteme GmbH. All rights reserved.
//********************************************************* 
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Runtime.InteropServices; // DllImport

namespace megsv86_cs
{
    class MEGSV64
    {
        // Version of this DLL
        public const int FILEVER_H = 1;
        public const int FILEVER_L = 27;

        // Return-codes for Simple errorcode
        public const int GSV_OK = 0; // no error, without signalling information
        public const int GSV_ERROR = -1; // an error occurred within the request
        public const int GSV_TRUE = 1; // no error, with signalling information

        // Device Model constants
        public const int GSV6 = 6;
        public const int GSV8 = 8;

        // Constant values used by GSV86*actEx
        public const int CONST_BAUDRATE = 115200;
        public const int CONST_BUFSIZE = 48000;

        // Flags for parameter "flags" in GSV86activateExtended
        public const int ACTEX_FLAG_HANDSHAKE = 4; // enables HW-handshake in GSV86activateExtended
        public const int ACTEX_FLAG_WAIT_EXTENDED = 0x100; // waits longer for device-answer in GSV86activateExtended
        public const int ACTEX_FLAG_STOP_TX = 0x200; // stops continious data transmission in GSV86activateExtended

        // Constants for GSV86getLastProtocollError and GSV86getLastErrorText
        public const int ERRTEXT_SIZE = 256; // Maximum size of error text
        public const uint ERR_MASK_ALL = 0xFC000000; // Mask for error code types
        public const int OWN_ERR_MASK = 0x20000000; // Mask, which can be or-ed with one of the well-known Windows Systen Error Code , see: MSDN:GetLastError()
        public const int ERR_MSK_DEVICE = 0x38000000; // Error given by GSV-8 device, errorcode in Bits<7:0>, see GSV-8 manual
                                                      // This application Errorcodes. Retrieve with GSV86getLastProtocollError. See file Errorcodes.h

        // Hardware properties of GSV-8 device
        public const int IN_CHAN_NO = 8; // Number of analogue input channels
        public const int SIX_AXIS_CHAN_NUM = 6; // Number of channels of the six-axis sensor
        public const int NUM_INTYPES = 6; // Number of switchable input types

        // Object Mapping, as for GSV86getValObjectInfo
        public const int VALOBJ_NUM_MAX = 16; // Arraysize for ScaleFactors and ObjMapping

        // 1. Value type in  Bits<15:8> of ObjMapping
        public const int VALTYPE_NORMAL = 0; // actual measuring value
        public const int VALTYPE_MAXVAL = 1; // Maximum value
        public const int VALTYPE_MINVAL = 2; // Minimum value

        // 2. Physical type in Bits<23:16> of ObjMapping
        public const int VAL_PHYS_TYPE_NOTDEF = 0; // no physical type defined (unknown)
        public const int VAL_PHYS_TYPE_FORCE_X = 1; // Force in X-direction
        public const int VAL_PHYS_TYPE_FORCE_Y = 2; // Force in Y-direction
        public const int VAL_PHYS_TYPE_FORCE_Z = 3; // Force in Z-direction
        public const int VAL_PHYS_TYPE_TORQUE_X = 4; // Torque-moment in X-direction
        public const int VAL_PHYS_TYPE_TORQUE_Y = 5; // Torque-moment in Y-direction
        public const int VAL_PHYS_TYPE_TORQUE_Z = 6; // Torque-moment in Z-direction
        public const int VAL_PHYS_TYPE_RAW = 0x10; // Raw value of six-axis sensor (before calculation)
        public const int VAL_PHYS_TYPE_TEMP = 0x20; // Temperature

        /* 	Struct definitition for convenience. ObjMapping parameter can be type-casted to this
        after return from GSV86getValObjectInfo.
        Must be packed to total size of 32 Bits, so better use with C or C++ only. */

        public class OBJECT_MAPPING
        {
            public byte ChannelNo; // Channel-Numbers from 1 to 8
            public byte ValueType; // Value type as defined above
            public byte PhysicType; // Physical type as defined above
            public byte reserved; // not yet used
        }

        // 3. DataType enumeration definition for GSV86getValObjectInfo,GSV86getInterfaceIdent, GSV86setValDataType parameter
        public const int DATATYP_INT16 = 1; // Datatype sent by GSV-8 is 16-Bit value in binary offset format
        public const int DATATYP_INT24 = 2; // Datatype sent by GSV-8 is 24-Bit value in binary offset format
        public const int DATATYP_FLOAT = 3; // Datatype sent by GSV-8 is 32-Bit Float (scaled in physical units)

        // Parameter of GSV86getInterfaceIdentity
        public const int SET_STOP_MV_TX = 1; // StopPermTX parameter: Disable permanent measuring value transmission
        public const int SET_START_MV_TX = 2; // StopPermTX parameter: Enable permanent measuring value transmission

        // Constants for GSV86getMode and GSV86setMode Flag-Parameter Mode
        public const int SIX_AXIS_SENSOR_ACTIVE = 0x1; // Six-axis sensor (=FT-sensor) calculation enabled
        public const int SIX_AXIS_SENSOR_ACT_GSV6 = 0x400; // same, but for GSV-6
        public const int ANALOG_FILTER_AUTO = 0x2; // Analog input filter's cut-off frequency set automatically according to data rate
        public const int MODE_MAXIMUM = 0x4; // determine maximum- and minimum value
        public const int MODE_MAXIMUM_GSV6 = 0x80; // same, but for GSV-6
        public const int MODE_NOISECUT = 0x8; // if =1: values below noiseCutThreshold are set to 0
        public const int ALL_WRITES_BLOCKED = 0x80; // all write / set commands are blocked

        // Constants for GSV86getTXmode and GSV86setTXmode Flag-Parameter
        public const int TX_OFF_VOLATILE = 0x1; // permanent measured value transmission actually off (not stored in EEPROM)
        public const int TX_OFF_NONVOLAT = 0x2; // permanent measured value transmission permanent off (stored in EEPROM)
        public const int TX_MAXVALUE = 0x4; // values in measured vals frame are maximum values
        public const int TX_MINVALUE = 0x8; // values in measured vals frame are minimum values
        public const int THRESH_VALTYPE_FLOAT = 0x10; // values for noiseCutThreshold and DoutThreshold are stored as float

        // Type enums (constants) for GSV86readFTsensorCalValue
        public const int SENSORCAL_TYP_SERNO = 0; // Sensor serial number. Type-exeption: u32 not float!
        public const int SENSORCAL_TYP_MATRIX_NORM = 1; // scaling of values in matrix (=1 for normalized matrix values)
        public const int SENSORCAL_TYP_MATRIX = 2; // 6-axis sensor coefficients 6x6 =36 matrix
        public const int SENSORCAL_TYP_OFFSET = 3; // mechanical offsets, 3 values Lx,Ly,Lz
        public const int SENSORCAL_TYP_MAXVAL = 4; // sensor nominal maximum values, 6vals Fx,Fy,Fz,Mx,My,Mz
        public const int SENSORCAL_TYP_INSENS = 5; // Input sensitivity of amp wich matrix vals were obtained (=1 for normalized matrix values)
        public const int SENSORCAL_TYP_ZEROVAL = 6; // Zero signal values of unloaded sensor
        public const double SENSORCAL_MATRIX_NORM_DEF = 5.0; // default value for SENSORCAL_TYP_MATRIX_NORM
        public const double SENSORCAL_INSENS_DEF = 2.0; // default value for SENSORCAL_TYP_INSENS
        public const double SENSORCAL_ZEROVAL_DEF = 0.0; // default value for SENSORCAL_TYP_ZEROVAL
        public const int SENSORCAL_OFFSET_DEF = 0; // default value for SENSORCAL_TYP_OFFSET
        public const int SENSORCAL_TYP_TEDS_MEM = 0x80; // Memory type-offset for TEDS memory (reserved)
        public const int SENSORCAL_TYP_EEPROM_MEM = 0; // Memory type-offset for EEPROM memory
        public const int SENSORCAL_VECTOR_SIZE = 6;
        public const int SENSORCAL_MATRIX_SIZE = 36;
        public const int SENSORCAL_OFFSET_SIZE = 3;
        public const int SENSORCAL_MAXVAL_SIZE = 6;

        // serial number constants
        public const int SER_NO_SIZE = 8;
        public const int SER_NO_CHAR_SIZE = SER_NO_SIZE + 1;    // inclusive termination
        public const int SERNO_MIN = 1;
        public const int SERNO_MAX = 99999999;

        // Constants for GSV86getInTypeRange, GSV86getAllInTypesRanges and GSV86setInType
        public const int MAX_INPUT_TYPES_NUM = 10; // Maximum possible number of input types.

        // (bigger than max. existent type-num, for future compatibiliy)
        public const int INTYP_BRIDGE_US875 = 0; // Input-Type Bridge at Vexcitation= 8.75V
        public const int INTYP_BRIDGE_US5 = 1; // Input-Type Bridge at Vexcitation= 5V
        public const int INTYP_BRIDGE_US25 = 2; // Input-Type Bridge at Vexcitation= 2.5V
        public const int INTYP_SE10 = 3; // Input-Type single-endend +-10V
        public const int INTYP_PT1000 = 4; // Input-Type Temperature-Sensor PT1000
        public const int INTYP_TEMP_K = 5; // Input-Type Temperature-Sensor Type-K (reserved)
        public const int INTYP_COUNTER = 6; // Input-Type counter / QEI / frequency

        // Constants for GSV86getUnitText and GSV86setUnitText
        public const int UNIT_WIDTH = 9; // maximum width of unit text array incl. termination

        // Bits <7:0> of Code parameter. Can be ORed with Bit<23:8> constants
        public const int ACTIVE_UNIT_ANY = 0x0; // GSV86getUnitText: Read active unit. GSV86setUnitText: Set and activate any unit
        public const int USER_UNIT_1 = 0x1; // GSV86getUnitText: Read user defined unit string 1. 
                                            // GSV86setUnitText: Set And activate user defined unit String 1.
        public const int USER_UNIT_2 = 0x2; // GSV86getUnitText: Read user defined unit string 2. 
                                            // GSV86setUnitText: Set And activate user defined unit String 2.
        public const int SET_FIXED_UNIT = 0x10; // GSV86setUnitText: Set and activate fixed unit (UnitNo set)
        public const int WRITE_USER_UNIT_1 = 0x11; // GSV86setUnitText: Write user unit string 1 (not activated)
        public const int WRITE_USER_UNIT_2 = 0x12; // GSV86setUnitText: Write user unit string 2 (not activated)

        // Bits <23:0> of Code parameter. Can be ORed with Bit<7:0> constants 
        public const int ANSI_CODEPAGE = 0x0; // ANSI 8-Bit coded
        public const int ASCII_ONLY = 0x100; // Use ASCII 7-Bit only
        public const int DOS_CODEPAGE_437 = 0x1B500; // DOS/Windows Codepage 437
        public const int WIN_CODEPAGE_1252 = 0x4E400; // Windows Codepage 1252

        // Constants for digital filters GSV86getDfilterOnOff and GSV86setDfilterOnOff
        public const int FILT_TYPE_IIR = 0; // Mask or for IIR-Type.
        public const int FILT_TYPE_FIR = 0x80; // Mask or for FIR-Type. See function description for details
        public const int FILT_TYPE_IIR_LP = 0x4; // IIR low pass, 4th order
        public const int FILT_TYPE_IIR_HP = 0x14; // IIR high pass, 4th order
        public const int FILT_TYPE_IIR_BP = 0x24; // IIR band pass, 4th order
        public const int FILT_TYPE_IIR_BS = 0x34; // IIR band stop, 4th order
        public const int FILT_TYPE_UNCONFIG = 0xFF; // Filter has nor been configured. v.1.18: Changed 
        public const int FILT_ORDER_MSK = 0xF; // Mask for filter order
        public const int FILT_ORDER_IIR = 4; // As by now, filter order is constant for IIR filter
        public const int FILT_MAXORDER_FIR = 14; // Maximum FIR filter Order
        public const int FILT_MINORDER_FIR = 4; // Minimum FIR filter Order
        public const int FILT_COEFFNUM_FIR = 8; // Because a0=a7, a1=a6... coeff. number is order/2
        public const int FILT_CHARACT_MSK = 0x70; // Mask for filter characteristic (LP,HP,BP,BS)
        public const int FILT_CHARACT_LP = 0x0; // low pass
        public const int FILT_CHARACT_HP = 0x10; // high pass
        public const int FILT_CHARACT_BP = 0x20; // band pass
        public const int FILT_CHARACT_BS = 0x30; // band stop
        public const int FILT_CHARACT_COMB = 0x40; // comb filter (FIR only)
        public const int FILT_IDX_MSK_B_COEFF = 0x10; // Mask for IIR index parameter for b coefficients
        public const float FILT_FCUT_RATIO_MAX = 0.5F; // maximum ratio for Fcutoff/Fs

        // Constants for GSV86simulateDfilter. May be ORed with FILT_TYPE_FIR
        public const int SIMUL_DFILT_FREQ_RESPONSE = 1; // simulate frequency response (frequency domain)
        public const int SIMUL_DFILT_STEP_RESPONSE = 2; // simulate step response (time domain)

        // Constants for analogue filter
        public const int FDATA_MAX_FG28HZ = 80; // With SetAnalogFilterAuto: Below this data rate filter is set to 28Hz cutoff
        public const int FDATA_MAX_FG1KHZ = 3000; // With SetAnalogFilterAuto: Below this data rate filter is set to 850Hz cutoff
        public const int ANAFILT_LOW = 28; // lowest cut-off frequency of analogue input filter (switchable)
        public const int ANAFILT_MID = 850; // middle cut-off frequency of analogue input filter (switchable)
        public const int ANAFILT_HIGH = 12000; // highest cut-off frequency of analogue input filter (fixed)

        // Constants for analog output
        public const int AOUT_TYPE_0_10V = 0; // 0..10V
        public const int AOUT_TYPE_10_10V = 1; // -10..10V
        public const int AOUT_TYPE_0_5V = 2; // 0..5V
        public const int AOUT_TYPE_5_5V = 3; // -5..5V
        public const int AOUT_TYPE_4_20A = 4; // 4..20mA
        public const int AOUT_TYPE_0_20A = 6; // 0..20mA

        //Mode: Flag value as follows:
        public const int AOUT_ACTIVE_M_VALUES = 0; // Output is on and follows the corresponding measuring value input.
        public const int AOUT_MODE_DIRECT = 1; // Output is on, but does NOT react on measuring values,
                                               // but Is direcly settable with GSV86writeAoutDirect
        public const int AOUT_MODE_OFF = 2; // Output is off and high-impedance.
        public const int AOUT_IX_MAX = 8; // 8 Analogue Voltage/current outputs.
        public const int AOUT_SCALE_IX_MAX = 9; // No. 9 for special-version (freq. output)

        // Constants for GSV86getEquipment
        public const int HAS_ADC = 0x1;
        public const int HAS_ETHERCAT = 0x2;
        public const int HAS_LCD = 0x4;
        public const int HAS_TEDS = 0x8;
        public const int HAS_DIGI_IO = 0x10;
        public const int HAS_ANALOG_OUT = 0x40;
        public const int HAS_SERIAL = 0x80;
        public const int HAS_FREQ_OUT = 0x100;
        public const int HAS_AIN_MCU = 0x200;
        public const int HAS_SIXAXIS = 0x400;
        public const int HAS_CANOPEN = 0x800;

        // Constants for DIOtype
        public const int DIO_IN_GENERALPURPOSE = 0x4;
        public const int DIO_IN_SYNC_SLAVE = 0x2;
        public const int DIO_IN_QEI_ANY = 0x8;
        public const int DIO_IN_TARE_SINGLE = 0x10;
        public const int DIO_IN_TARE_ALL = 0x20;
        public const int DIO_IN_RESET_MAXMIN = 0x40;
        public const int DIO_IN_TRIG_SEND_VAL = 0x80;
        public const int DIO_IN_TRIG_SEND_MAXVAL = 0x100;
        public const int DIO_IN_TRIG_SEND_MINVAL = 0x200;
        public const int DIO_IN_TRIG_SEND_AVGVAL = 0x400;
        public const int DIO_IN_TRIG_SEND_VAL_WHILE_HI = 0x800;
        public const int DIO_OUT_GENERALPURPOSE = 0x1000;
        public const int DIO_THRESHOLD_WINDOWCOMP_MASK = 0x2000;
        public const int DIO_OUT_THRESHOLD_MAXVAL = 0x14000;
        public const int DIO_OUT_THRESHOLD_MINVAL = 0x18000;
        public const int DIO_OUT_THRESHOLD_ANYVAL = 0x10000;
        public const int DIO_OUT_SYNC_MASTER = 0x20000;
        public const int DIO_INVERT_MASK = 0x800000;

        // Constants for GSV86readCANsettings / GSV86setCANsettings
        public const int CAN_SER_CMD_ID = 0;
        public const int CAN_SER_ANSW_ID = 1;
        public const int CAN_SER_MV_ID = 2;
        public const int CAN_SER_MULTI_ID = 3;
        public const int CAN_BAUDRATE = 4;
        public const int CAN_FLAGS = 5;
        public const int CANOPEN_NODEID = 6;
        public const int CAN_FLAGMASK_ONOFF = 1;
        public const int CAN_FLAGVAL_ON = 0;
        public const int CAN_FLAGVAL_OFF = 1;
        public const int CAN_FLAGMASK_PROT = 2;
        public const int CAN_FLAGVAL_CANOPEN = 2;
        public const int CAN_FLAGVAL_CANSER = 0;

        // Constant macros for use with Communication interface descriptor functions
        // GSV86readInterfaceSetting
        // GSV86writeInterfaceSetting
        // GSV86readBasicInterfSettings
        // GSV86readAllInterfSettings
        // Interface Physical Layer protocol enumeration
        public const int INTF_PHY_TYP_V24 = 1; // RS232 with V24 voltage levels
        public const int INTF_PHY_TYP_232TTL = 2; // RS232 with 3.3V / 0V voltage levels
        public const int INTF_PHY_TYP_USB = 3; // USB (with CDC)
        public const int INTF_PHY_TYP_CAN = 4; // CAN bus
        public const int INTF_PHY_TYP_ETH = 5; // 100BaseT "Ethernet"

        // Interface Application Layer protocol enumeration
        public const long INTF_APP_TYP_GSV68 = 0x1; // ME propretary "GSV6/8" interface, used by this DLL
        public const long INTF_APP_TYP_CANOPEN = 0x2; // CANopen
        public const long INTF_APP_TYP_ETCOE = 0x3; // EtherCAT CoE
        public const long INTF_APP_TYP_MONI = 0x4; // Text-based commands "Monitor", GSV-6 only

        // The flags-value of basic interface setting
        public const int INTF_T1_FLG_ACT = 1; // Flag (-mask) indicating that interface is enabled
        public const int INTF_T1_FLG_RXACT = 1; // Flag (-mask) indicating that receiving commands is enabled
        public const int INTF_T1_FLG_TXACT = 2; // Flag (-mask) indicating that transmitting data is enabled
        public const int INTF_T1_FLG_TXRX_ACT = 3; // Flag (-mask) indicating that transmit and receive is enabled
        public const int INTF_T1_FLG_WRACC = 4; // Flag (-mask) indicating that write commands are allowed
        public const int INTF_T1_MV_BINOFS = 8; // Flag (-mask) indicating that integer measuring values have binary offset formal (sign bit inverted)
        public const int INTF_T1_FLG_SWITCHABLE = 0x10000; // Flag (-mask) indicating that interface can be disabled (then enabled)
        public const int INTF_T1_FLG_ADRESSABLE = 0x20000; // Flag (-mask) indicating that interface uses device or service address/ID
        public const int INTF_T1_FLG_ADRES_CHANGE = 0x40000; // Flag (-mask) indicating that device / service address/ID can be changes by command interface

        // Data/Setting type 2 meaning enumeration
        public const int INTF_DSCTYP_FLAGS = 0; // Flag value of basic interface setting
        public const int INTF_DSCTYP_FLG_SETMSK = 1; // Mask for setting a bit in the Flag value of the basic setting =1:Flag at this pos. can be set
        public const int INTF_DSCTYP_FLG_CLRMSK = 2; // Mask for clearing a bit in the Flag value of the basic setting =1:Flag at this pos. can be cleared
        public const int INTF_DSCTYP_MUTEXCLUSIV = 3; // Ordinal number of other interfaces, that this exist in a mutual exclusive manner. Upto 4 in 4 Bytes, beginning with LSByte
        public const int INTF_DSCTYP_ACTBAUD = 4; // Baud Rate actually set, in Bits/s
        public const int INTF_DSCTYP_BAUD_AVAIL = 5; // Baud rate existent, in Bits/s
        public const int INTF_DSCTYP_NUM_SERVID = 6; // Number of IDs of device or servicve, used with field bus interfaces
        public const int INTF_DSCTYP_SERVID_CAN_IN = 7; // CAN-ID for Host->Device commands, used with "GSV68 CAN-serial"
        public const int INTF_DSCTYP_SERVID_CAN_OUT = 8; // CAN-ID for Device->Host command answers, used with "GSV68 CAN-serial"
        public const int INTF_DSCTYP_SERVID_CAN_CV = 9; // CAN-ID for Device->Host measuring value frames, used with "GSV68 CAN-serial"
        public const int INTF_DSCTYP_SERVID_CAN_CAST = 10; // CAN-ID for Host->Devices multicast messages, used with "GSV68 CAN-serial"
        public const int INTF_DSCTYP_SERVID_CANO_NODE = 11; // CANopen Node-ID, used with CANopen aplication protocol
        public const int INTF_DSCTYP_DEVSTATE = 16; // Device state, used with fieldbus application protocols
        public const int INTF_FB_STAT_OFF = 0; // Data-value for INTF_DSCTYP_DEVSTATE if Interface is off
        public const int INTF_FB_STAT_INIT = 2; // Data-value for INTF_DSCTYP_DEVSTATE if device is in init or stopped state
        public const int INTF_FB_STAT_PO = 4; // Data-value for INTF_DSCTYP_DEVSTATE if device is in pre-operational state
        public const int INTF_FB_STAT_SO = 8; // Data-value for INTF_DSCTYP_DEVSTATE if device is in safe operational state (EtherCAT)
        public const int INTF_FB_STAT_O = 12; // Data-value for INTF_DSCTYP_DEVSTATE if device is in operational state
        public const int INTF_DSCTYP_CANO_TX_TIME = 17; // CANopen Transmission Type (Bits<31:24>) and Event timer (Bits<15:0>)
        public const int INTF_DSCTYP_CANO_INH_HB = 18; // CANopen Inhibit time (Bits<31:16>) and Heartbeat time (Bits<15:0>)
        public const int INTFT2_WR = 0x80; // Flag value indicating that a value is writable. ORed with data type enum

        ///////////////////////////////////////////////////////////////////////////////////
        //// Exported functions Prototypes
        //--------------------------------------------------------------------------------
        //////////////////////////////////////////////////////////////////////////////////

        // https://docs.microsoft.com/en-us/dotnet/api/system.runtime.interopservices.callingconvention?view=netframework-4.7.2
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86dllVersion();
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86actExt(int ComNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86activateExtended(int ComNo, uint Bitrate, uint BufSize, uint flags);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getLastProtocollError(int ComNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getLastDeviceError(int ComNo, int Async);
        [DllImport("MEGSV86x64.dll", CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getLastErrorText(int ComNo, StringBuilder ErrText);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86clearDLLbuffer(int ComNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getInterfaceIdentity(int ComNo, int StopPermTX, ref int NumOfIntfDescr, ref int ThisInterfNo, ref int WriteProtect, ref int MValDataType, ref int MValPermanenTX, ref int NumObjInMVframe, ref int ThisDeviceModel, ref int ThisProtocol);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86clearDeviceBuf(int ComNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86received(int ComNo, int Chan);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86read(int ComNo, int Chan, ref double @out);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86readMultiple(int ComNo, int Chan, ref double @out, int count, ref int valsread, ref int ErrFlags);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86startTX(int ComNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86stopTX(int ComNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86release(int ComNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getSerialNo(int ComNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getTXmode(int ComNo, int Index);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setTXmode(int ComNo, int Index, uint TXmode);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86isValTXpermanent(int ComNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getWriteAccess(int ComNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setValDataType(int ComNo, int Datatype);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86firmwareVersion(int ComNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86readUserScale(int ComNo, int Chan, ref double Norm);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86writeUserScale(int ComNo, int Chan, double Norm);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getMode(int ComNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setMode(int ComNo, uint Mode);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getFTsensorActive(int ComNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setFTsensorActive(int ComNo, int OnOff);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getModeAfilterAuto(int ComNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setModeAfilterAuto(int ComNo, int OnOff);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getModeNoiseCut(int ComNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setModeNoiseCut(int ComNo, int OnOff);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getModeMaxMin(int ComNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setModeMaxMin(int ComNo, int OnOff);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getValObjectInfo(int ComNo, ref double ScaleFactors, ref ulong ObjMapping, ref int DataType);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getValMapping(int ComNo, int Index);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setValMapping(int ComNo, int Index, uint ObjMapping);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setFTarrayToRead(int ComNo, int ArrNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86readFTsensorCalValue(int ComNo, int typ, int ix, ref double val);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86readFTsensorCalArray(int ComNo, int ArrNo, ref string SensorSerNo, ref double MatrixNorm, ref double InSens, ref double Matrix, ref double Offsets, ref double MaxVals, ref double Zvals);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86writeFTsensorCalArray(int ComNo, int ArrNo, string SensorSerNo, double MatrixNorm, double InSens, ref double Matrix, ref double Offsets, ref double MaxVals, ref double Zvals);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86writeFTsensorGeoOffsets(int ComNo, int ArrNo, ref double offsets);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86readFTsensorSerNo(int ComNo, int ArrNo, ref string SensorSerNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86writeFTsensorFromFile(int ComNo, int ArrNo, string DatFilePath);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getFTsensorCalArrayInfo(int ComNo, ref int MaxNumSupp, ref int ArrNumStored);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setFTsensorActiveCalArray(int ComNo, int ArrNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86eraseFTsensorCalArray(int ComNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern double GSV86getFrequency(int ComNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setFrequency(int ComNo, double frequency);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setZero(int ComNo, int Chan);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getInTypeRange(int ComNo, int Chan, ref double Range);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getAllInTypesRanges(int ComNo, int Chan, ref int InTypes, ref double Ranges);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setInType(int ComNo, int Chan, int InType);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86readUserOffset(int ComNo, int Chan, ref double Offset);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86writeUserOffset(int ComNo, int Chan, double Offset);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86loadSettings(int ComNo, int DataSetNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86storeSettings(int ComNo, int DataSetNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getAnalogOutType(int ComNo, int Chan, ref int Type);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setAnalogOutType(int ComNo, int Chan, int Type, int Mode);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86writeAnalogOutOffset(int ComNo, int Chan, double Offset);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86readAnalogOutOffset(int ComNo, int Chan, ref double Offset);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86readAnalogOutScale(int ComNo, int Chan, ref double Scale);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86writeAnalogOutScale(int ComNo, int Chan, double Scale);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86writeAoutDirect(int ComNo, int Chan, int Code);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getUnitNo(int ComNo, int Chan);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setUnitNo(int ComNo, int Chan, int UnitNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getUnitText(int ComNo, int Chan, int Code, ref string UnitText);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setUnitText(int ComNo, int Chan, int Code, ref string UnitText);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getDfilterOnOff(int ComNo, int Chan, int Type);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setDfilterOnOff(int ComNo, int Chan, int Type, int OnOff);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getDfilterInfo(int ComNo, int Chan, int TypeIn, ref int TypeOut, ref double CutOff);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getDfilterType(int ComNo, int Chan, int TypeIn);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getDfilterCoeff(int ComNo, int Chan, int Type, ref double Coeff, ref double CoeffB);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setDfilterParams(int ComNo, int Chan, int Type, ref double CutRatio, ref double Coeff, ref double CoeffB);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86calcSetDfilterParams(int ComNo, int Chan, int Type, double CutOff, double CutOffHi);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86simulateDfilter(int ComNo, int Analyze_Ftyp, double StartVal, double EndVal, double Fa, int Points, ref string Filepath);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern double GSV86readDeviceHours(int ComNo, int Index);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86writeDeviceHours(int ComNo, double Hours);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getNoiseCutThreshold(int ComNo, int Chan, ref double Thres);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setNoiseCutThreshold(int ComNo, int Chan, double Thres);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getSoftwareConfiguration(int ComNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setMeasValProperty(int ComNo, int PropType);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getValueError(int ComNo, int Ix, ref int ErrInfo, ref double ErrTime);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86eraseValueErrMemory(int ComNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86resetErrorState(int ComNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86readAnalogFilterCutOff(int ComNo, ref double CutOffFreq);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86writeAnalogFilterCutOff(int ComNo, double CutOffFreq);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86readZeroValue(int ComNo, int Chan);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86writeZeroValue(int ComNo, int Chan, int zero);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getIDlevel(int ComNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86triggerValue(int ComNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getMaxMinValue(int ComNo, int Chan, ref double MaxValue, ref double MinValue);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86clearMaxMinValue(int ComNo, int Chan);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getIsCmdAvailable(int ComNo, int CmdUp, int CmdLo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86switchBlocking(int ComNo, int OnOff);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86changePassword(int ComNo, ref string NewPW);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setPassword(int ComNo, ref string password);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getDIOdirection(int ComNo, int DIOgroup);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setDIOdirection(int ComNo, int DIOgroup, int Direction);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getDIOtype(int ComNo, int DIOno, ref int AssignedChan);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setDIOtype(int ComNo, int DIOno, int DIOtype, int AssignedChan);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getDIOlevel(int ComNo, int DIOno);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setDoutLevel(int ComNo, int DIOno, int DIOlevel);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getDoutThreshold(int ComNo, int DIOno, ref double ThresUp, ref double ThresDown);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setDoutThreshold(int ComNo, int DIOno, double ThresUp, double ThresDown);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getDoutInitLevel(int ComNo, int DIOno);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setDoutInitLevel(int ComNo, int DIOno, int DOInitLevel);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getDataRateRange(int ComNo, ref double DrateMax, ref double DrateMin);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86readCANsettings(int ComNo, int Index, ref int setting);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setCANsettings(int ComNo, int Index, int setting);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getCANonOff(int ComNo, ref int CANappProt);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setCANonOff(int ComNo, int OnOff);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86readInterfaceSetting(int ComNo, int Ix, ref int Next, ref uint Data, ref int ApplEnum, ref int Writable);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86readBasicInterfSettings(int ComNo, int ActIntf, ref int PhysEnums, ref int ApplEnums, ref int Flags);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86readAllInterfSettings(int ComNo, int IntNo, ref int IntfEnums, ref int Dtypes, ref int Data, ref int BdList, ref int BdNum);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86writeInterfaceSetting(int ComNo, int Ix, uint Data);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setInterfaceOnOff(int ComNo, int IntNo, int OnOff);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86writeInterfaceBaud(int ComNo, int IntNo, int Baud);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86readHWversion(int ComNo, ref int MainHW, ref int ExtHW);

        public const int TEDS_ANSW_IS_FLT = 1;
        public const int TEDS_IS_PACKED_CHR5 = 2;
        public const int TEDS_IS_DATE_DAYS = 4;
        public const int TEDS_ENTRY_HAS_ERROR = 0x80;
        public const int TEDS_ENTRY_NOT_EXIST = 0xFF;
        public const int TEDS_ENTRY_NOT_SET = 0xFE;
        public const int TEDS_ENTRY_INVALID = 0xFD;

        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86readTEDSentry(int ComNo, int Chan, int TemplID, int PropID, ref int Next, int No, ref uint Udata, ref double DblData, ref int Flags);
                
        public const int TEDSLISTFLG_BASICONLY = 1; // List Basic template only
        public const int TEDSLISTFLG_MAINONLY = 2; // List Main template only
        public const int TEDSLISTFLG_COLUMN_NUM_VAR = 4; // Empty field left away, so that column number is variable
        public const int TEDSLISTFLG_FILL_EMPTY_SPACE = 8; // Insert space in empty field
        public const int TEDSLISTFLG_PIPE_SEPARA = 16; // Field separator inside line is | otherwise TAB

        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86readFormattedTEDSList(int ComNo, int Chan, string TEDSfilePath, ref string ListOut, int ListSize, int Code, ref string ExtListOut);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86readTEDSrawData(int ComNo, int Chan, ref byte DataOut, int NumBytes, int StartByteAdr);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86writeTEDSrawData(int ComNo, int Chan, byte DataIn, int NumBits, int StartBitAdr);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getSensorPlugged(int ComNo, int Chan, ref int BridgeSensor, ref int TEDScapable);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getTEDSactive(int ComNo, int Chan);

        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86resetDevice(int ComNo);

        public const int COUNTER_PRESENT = 1;
        public const int COUNT_FREQ_ON = 6;
        public const int COUNTER_ENABLED = 2;
        public const int FREQ_SPEED_ENABLE = 4;
        public const int QEI_MODE_MASK = 0x18;
        public const int QEI_MODE_FREERUN = 0;
        public const int QEI_MODE_X1 = 0x8;
        public const int QEI_MODE_X2 = 0x10;
        public const int QEI_MODE_X4 = 0x18;
        public const int QEI_USE_HOME = 0x20;
        public const int COUNTER_NOSAT = 0x40;
        public const int STORE_COUNT_VAL = 0x80;

        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86getCounterFreqMode(int ComNo, int CntNo, int index);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86setCounterFreqMode(int ComNo, int CntNo, int index);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86BTinitConfig(int ComNo, ref int BGversion);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86BTexitConfig(int ComNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86BTgetMaxPower(int ComNo, int BTmode, ref int pwr);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86BTgetBTmode(int ComNo, ref int BTmode);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86BTsetBTmode(int ComNo, int BTmode);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86BTreadName(int ComNo, ref char Name);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86BTwriteName(int ComNo, ref char Name);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86BTreset(int ComNo, int ResetType);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86BTgetGSVonOff(int ComNo, ref int OnOff);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86BTsetGSVonOff(int ComNo, ref int OnOff);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86BTreadBatteryVoltage(int ComNo, ref double Voltage);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86BTsetDefault(int ComNo);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86BTgetInType(int ComNo, int chan, ref int val);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86BTsetInType(int ComNo, int chan, int val);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86BTsetDigitalOut(int ComNo, int DIOno, int OnOff);

        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86MEwriteCal(int ComNo, int Chan, int ix, double cal);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86readCal(int ComNo, int Chan, int ix, ref double cal);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int GSV86MEwriteInputRange(int ComNo, int Chan, int type, double range);
        [DllImport("MEGSV86x64.dll", CharSet = CharSet.Unicode, CallingConvention = CallingConvention.Cdecl)]
        public static extern int MEgsv86setSerialNo(int ComNo, int SerNo);
    }
}
